import 'package:aad_oauth/aad_oauth.dart';
import 'package:absenkuv3/Bloc/AuthBloc.dart';
import 'package:absenkuv3/Bloc/NavigatorBloc.dart';
import 'package:absenkuv3/Setting/oauth_config.dart';
import 'package:absenkuv3/UI/Template/style.dart';
import 'package:absenkuv3/UI/Template/templateButton.dart';
import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:absenkuv3/Bloc/SplashScreenBloc.dart';
import 'package:absenkuv3/Setting/value_image.dart';

class HalamanSplashScreen extends StatefulWidget {
  const HalamanSplashScreen({super.key});

  @override
  State<HalamanSplashScreen> createState() => _HalamanSplashScreenState();
}

class _HalamanSplashScreenState extends State<HalamanSplashScreen>
    with TickerProviderStateMixin {
  final GlobalKey<ScaffoldState> _scaffoldKey = GlobalKey<ScaffoldState>();

  late AnimationController _cardController;
  late Animation<Offset> _cardSlideAnimation;
  late Animation<double> _cardFadeAnimation;

  late AnimationController _logoController;
  late Animation<Offset> _logoSlideAnimation;
  late Animation<double> _logoFadeAnimation;

  bool _allowAnimation = true;
  late AadOAuth oauth;

  @override
  void initState() {
    super.initState();
    oauth = OAuthService.oauth;
    BlocProvider.of<SplashScreenBloc>(context).add(SplashStarted(context));

    _cardController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 800),
    );

    _cardSlideAnimation = Tween<Offset>(
      begin: const Offset(0, 1),
      end: Offset.zero,
    ).animate(CurvedAnimation(parent: _cardController, curve: Curves.easeOut));

    _cardFadeAnimation = Tween<double>(
      begin: 0,
      end: 1,
    ).animate(CurvedAnimation(parent: _cardController, curve: Curves.easeIn));

    _logoController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 600),
    );

    _logoSlideAnimation = Tween<Offset>(
      begin: Offset.zero,
      end: const Offset(0, -0.5),
    ).animate(CurvedAnimation(parent: _logoController, curve: Curves.easeOut));

    _logoFadeAnimation = Tween<double>(
      begin: 1,
      end: 0,
    ).animate(CurvedAnimation(parent: _logoController, curve: Curves.easeOut));
  }

  void startAnimation() {
    Future.delayed(const Duration(milliseconds: 1000), () {
      if (_allowAnimation) {
        _cardController.forward();
        Future.delayed(const Duration(milliseconds: 800), () async {
          await _logoController.forward();
          setState(() {});
        });
      }
    });
  }

  void _loginAndFetchUser() async {
    try {
      await oauth.login();
      final accessToken = await oauth.getAccessToken();
      print('Access Token: $accessToken');

      if (accessToken != null && accessToken.isNotEmpty) {
        BlocProvider.of<AuthBloc>(context)
            .add(LoginSSOEvent(accessToken, context, _scaffoldKey));
      } else {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Gagal mendapatkan access token.')),
        );
      }
    } catch (e) {
      print('Login error: $e');
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Login error: $e')),
      );
    }
  }

  @override
  void dispose() {
    _cardController.dispose();
    _logoController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final size = MediaQuery.of(context).size;

    return BlocListener<SplashScreenBloc, SplashState>(
      listener: (context, state) {
        if (state is SplashToDashboard) {
          _allowAnimation = false;
          BlocProvider.of<NavigatorBloc>(context)
              .add(NavigateRemoveToDashboard(context));
        } else if (state is SplashToLogin) {
          _allowAnimation = true;
          startAnimation();
        }
      },
      child: Scaffold(
        body: Stack(
          children: [
            // Background
            AnimatedBuilder(
              animation: _cardController,
              builder: (context, child) {
                final offsetY = Tween<double>(begin: 0, end: -size.height * 0.4)
                    .animate(CurvedAnimation(
                        parent: _cardController, curve: Curves.easeOut))
                    .value;

                return Transform.translate(
                  offset: Offset(0, offsetY),
                  child: SizedBox(
                    width: size.width,
                    height: size.height,
                    child: Image.asset(
                      ValueImage.bgSplashScreen,
                      fit: BoxFit.fill,
                    ),
                  ),
                );
              },
            ),

            // Logo besar (di tengah)
            
            AnimatedBuilder(
              animation: _logoController,
              builder: (context, child) {
                const logoSize = 300.0;
                return Align(
                  alignment: Alignment.center,
                  child: FractionalTranslation(
                    translation: Offset(0, _logoSlideAnimation.value.dy),
                    child: Opacity(
                      opacity: _logoFadeAnimation.value,
                      child: Container(
                        width: logoSize,
                        height: logoSize,
                        padding: const EdgeInsets.all(8),
                        child: Image.asset(
                          ValueImage.logoSplashScreen,
                        ),
                      ),
                    ),
                  ),
                );
              },
            ),

            // Card Login
            Align(
              alignment: Alignment.bottomCenter,
              child: SlideTransition(
                position: _cardSlideAnimation,
                child: FadeTransition(
                  opacity: _cardFadeAnimation,
                  child: Container(
                    width: double.infinity,
                    height: size.height * 0.5,
                    decoration: const BoxDecoration(
                      color: Colors.white,
                      borderRadius:
                          BorderRadius.vertical(top: Radius.circular(30)),
                      boxShadow: [
                        BoxShadow(
                          color: Colors.black26,
                          blurRadius: 10,
                          offset: Offset(0, -2),
                        ),
                      ],
                    ),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.stretch,
                      children: [
                        Column(
                          children: [
                            SizedBox(
                              height: 30,
                            ),
                            Center(
                              child: Image.asset(
                                ValueImage.logoCardLogin,
                                width: 300,
                                height: 150,
                              ),
                            ),
                            const SizedBox(height: 40),
                            Text(
                              "Selamat Datang",
                              style: TextStyle(
                                  fontWeight: FontWeight.bold, fontSize: 25),
                            ),
                            const SizedBox(height: 20),
                            Text("Login Akunmu menggunakan Azure",
                                style: TextStyle(fontSize: 20)),
                            Container(
                              margin: EdgeInsets.all(25),
                              child: TemplateButton(
                                onClick: () async {
                                  _loginAndFetchUser();
                                },
                                label: "Login Azure",
                                color: CustomStyle.color1,
                                fontSize: CustomStyle.fontSizeNormal,
                                paddingTopBottom: 8,
                                expand: true,
                                borderColor: Colors.white,
                                radius: 24,
                                heightButton: 42,
                              ),
                            ),
                          ],
                        ),
                      ],
                    ),
                  ),
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }
}
